/*----------------------------------------------------------------------------
 * Name     : main.c
 * Purpose  : BC3602 test 
 * Note(s)  : 
 *----------------------------------------------------------------------------
 * This file is part of the uVision/ARM development tools.
 * This software may only be used under the terms of a valid, current,
 * end user licence from KEIL for a compatible version of KEIL software
 * development tools. Nothing else gives you the right to use this software.
 *
 * This software is supplied "AS IS" without warranties of any kind.
 *
 * Copyright (c) 2012 Keil - An ARM Company. All rights reserved.
 *----------------------------------------------------------------------------*/
#include  <stdio.h>
#include  <string.h>
#include  "BC3602.h"
#include	"timer.h"
#include	"main.h"

#define  _IRQ_ENABLE_   (1)
#define _IRQ_LINE_  (1)		
//       		 BC3602 IRQ for GPIO1/2/3/4
//          <0=> GPIO1
//          <1=> GPIO2
//          <2=> GPIO3
//          <3=> GPIO4
#define _IRQ_POLARITY_ (0)
#define  _LIRC_ENABLE_   (0)
#define  _BC3602_REGS_IRQ1_   (0x04)
#define  _BC3602_REGS_ARK1_   (82)
#define  _BC3602_REGS_ARK2_   (20)

#if ( _IRQ_LINE_ == _SDO_LINE_ )
   #error "GPIO function issue : SDO and IRQ can not be the same GPIO!"
#endif

bit_type Flag0;
#if (_SPI_ARCHITECTURE_ == 0)
static void  BC3602_serial_output(u8 data);
static u8    BC3602_serial_input(void);
#endif
void syncword_calc(u16 sel, u32 lap, u8 *pIdcode);

enum
{
	_TX_SEL_ = 0,
	_RX_SEL_
};

uc8   rf_power_value[][4] = 
{
	//   0,   5,  10,  13
	{ 0x06,0x08,0x0B,0x0F },         				//315  ,0502 fine tune result  
  { 0x04,0x06,0x09,0x0D },                //433  ,0502 fine tune result
  { 0x04,0x06,0x09,0x0D },         				//470  ,0502 fine tune result          
  { 0x06,0x08,0x0B,0x0E },               	//868  ,0502 fine tune result          
  { 0x06,0x08,0x0B,0x0E }          				//915  ,0502 fine tune result 
};

//uc8		margin_len[] = { 4, 8, 16, 32};
uc8		rx_preamble_value[] = { 0x00,0x01,0x02,0x03 };
uc8   sync_width_value[] = { 0x04,0x08,0x0c };											
uc8  	sync_word_address[] = { 0x47,0x5C,0x58,0xCC,0x73,0x34,0x5E,0x72};
u8		sync_word_buff[10];
uc8   pn9_data[] = 
{
  0xFF,0x83,0xDF,0x17,0x32,0x09,0x4E,0xD1, 
	0xE7,0xCD,0x8A,0x91,0xC6,0xD5,0xC4,0xC4,
	0x40,0x21,0x18,0x4E,0x55,0x86,0xF4,0xDC,
	0x8A,0x15,0xA7,0xEC,0x92,0xDF,0x93,0x53,
	0x30,0x18,0xCA,0x34,0xBF,0xA2,0xC7,0x59,
	0x67,0x8F,0xBA,0x0D,0x6D,0xD8,0x2D,0x7D,
	0x54,0x0A,0x57,0x97,0x70,0x39,0xD2,0x7A,
	0xEA,0x24,0x33,0x85,0xED,0x9A,0x1D,0xE0,
};



#define	_ANALOG_NUM_	8
// Bank2
//#if ( _RF_BAND_VALUE_ == 0)   /* 315M band */
u8   Analog_RegisterTable315[][2] = 
{	
	{0x26,0x03},
	{0x27,0x88},
	{0x28,0xA3},
	{0x2D,0x16},
	{0x2E,0x64},
	{0x30,0x00},
	{0x31,0x64},
	{0x34,0xBC},
	{0x3A,0x94}
};   

//#if ( _RF_BAND_VALUE_ == 1)   /* 433M band */
u8   Analog_RegisterTable433[][2] =
{  
	{0x26,0x03},
	{0x27,0x88},
	{0x28,0xAA},
	{0x2D,0x16},
	{0x2E,0x64},
	{0x30,0x00},
	{0x31,0x64},
	{0x34,0xBC},
	{0x3A,0x94}
};   

//#if ( _RF_BAND_VALUE_ == 2)   /* 470~510M band */
u8   Analog_RegisterTable470[][2] = 
{
	{0x26,0x03},
	{0x27,0x88},
	{0x28,0xAA},
	{0x2D,0x16},
	{0x2E,0x64},
	{0x30,0x00},
	{0x31,0x64},
	{0x34,0xBC},
	{0x3A,0x94}
};   

//#if ( _RF_BAND_VALUE_ == 3)   /* 868M/915M band */
u8   Analog_RegisterTable868[][2] =  						
{  	
	{0x26,0x03},
	{0x27,0x88},
	{0x28,0xAA},
	{0x2D,0x16},
	{0x2E,0x74},
	{0x30,0x00},
	{0x31,0x64},
	{0x34,0x9C},
	{0x3A,0x94}
};  
   
// Bank2
uc8 *Analog_Register_index[4] =
{
   (uc8 *)&Analog_RegisterTable315,
   (uc8 *)&Analog_RegisterTable433,
   (uc8 *)&Analog_RegisterTable470,
   (uc8 *)&Analog_RegisterTable868,
};    

/* Crystal = 16MHz */ 

u32 data_rate_Table16[] =
{
	2000,5000,10000,25000,50000,125000,250000
};

uc8 MOD_RegisterTable16[][8]=
{
 /* MOD1 MOD2 MOD3     Crystal 16MHz    */
   {0xF9,0x60,0x66},   /*data rate = 2K */	//h=8
   {0x63,0x60,0x66},   /*data rate = 5K */
   {0x31,0x60,0x66},   /*data rate = 10K */
   {0x13,0x60,0x66},   /*data rate = 25K */ 	//h=4	DM1 & DM2 different than Excel 
   {0x09,0x60,0x66},   /*data rate = 50K */	//h=0.75
   {0x03,0x90,0x9A},   /*data rate = 125K */ 
   {0x01,0x90,0x9A}    /*data rate = 250K */
};

#if ((DEFAULT_RX_Preamble ==_RX_Preamble_0B_) || (DEFAULT_RX_Preamble ==_RX_Preamble_1B_))
uc8 DM_RegisterTable16[][8]=
{
 /*  DM1  DM2  DM3  DM4  DM5            DM8     Crystal 16MHz    */
   {0x31,0x09,0xE6,0x08,0x1F,0x00,0x00,0x05},   /*data rate = 2K */
   {0x13,0x09,0xE6,0x08,0x1F,0x00,0x00,0x0D},   /*data rate = 5K */
   {0x09,0x09,0xE6,0x08,0x1F,0x00,0x00,0x1A},   /*data rate = 10K */
   {0x07,0x04,0xE6,0x08,0x1F,0x00,0x00,0x20},   /*data rate = 25K */
   {0x13,0x00,0xE0,0x08,0x3A,0x00,0x00,0x0D},   /*data rate = 50K */
   {0x07,0x00,0xE0,0x08,0x3A,0x00,0x00,0x20},   /*data rate = 125K */
   {0x03,0x00,0xE0,0x08,0x3A,0x00,0x00,0x40}    /*data rate = 250K */
};
#elif (DEFAULT_RX_Preamble ==_RX_Preamble_2B_)
uc8 DM_RegisterTable16[][8]=
{
 /*  DM1  DM2  DM3  DM4  DM5            DM8     Crystal 16MHz    */
   {0x31,0x49,0xE6,0x08,0x1A,0x00,0x00,0x05},   /*data rate = 2K */
   {0x13,0x49,0xE6,0x08,0x1A,0x00,0x00,0x0D},   /*data rate = 5K */
   {0x09,0x49,0xE6,0x08,0x1A,0x00,0x00,0x1A},   /*data rate = 10K */
   {0x07,0x44,0xE6,0x08,0x1A,0x00,0x00,0x20},   /*data rate = 25K */
   {0x13,0x40,0xE0,0x08,0x30,0x00,0x00,0x0D},   /*data rate = 50K */
   {0x07,0x40,0xE0,0x08,0x30,0x00,0x00,0x20},   /*data rate = 125K */
   {0x03,0x40,0xE0,0x08,0x30,0x00,0x00,0x40}    /*data rate = 250K */
};
#elif (DEFAULT_RX_Preamble ==_RX_Preamble_4B_)
uc8 DM_RegisterTable16[][8]=
{
 /*  DM1  DM2  DM3  DM4  DM5            DM8     Crystal 16MHz    */
   {0x31,0xC9,0xE6,0x48,0x1A,0x00,0x00,0x05},   /*data rate = 2K */
   {0x13,0xC9,0xE6,0x48,0x1A,0x00,0x00,0x0D},   /*data rate = 5K */
   {0x09,0xC9,0xE6,0x48,0x1A,0x00,0x00,0x1A},   /*data rate = 10K */
   {0x07,0xC4,0xE6,0x48,0x1A,0x00,0x00,0x20},   /*data rate = 25K */
   {0x13,0xC0,0xE0,0x48,0x30,0x00,0x00,0x0D},   /*data rate = 50K */
   {0x07,0xC0,0xE0,0x48,0x30,0x00,0x00,0x20},   /*data rate = 125K */
   {0x03,0xC0,0xE0,0x48,0x30,0x00,0x00,0x40}    /*data rate = 250K */
};
#endif

uc16 FCF_RegisterTable16[][10] = 
{
//	FCF1		FCF3,2, 5,4		7,6	 9,8	  11,10	13,12	 15,14  17,16  19,18		
/*     SFR    FSC   CB12   CB13   CA12   CA13   CB22   CB23   CA22   CA23  Crystal 16MHz */   
   {0x0036,0x0021,0x0000,0x0000,0x0302,0x0000,0x0000,0x0000,0x0000,0x0000 }, /* data rate=2K */ 	
   {0x0036,0x0052,0x0000,0x0000,0x0302,0x0000,0x0000,0x0000,0x0000,0x0000 }, /* data rate=5K */	
   {0x0016,0x00A4,0x0000,0x0000,0x0310,0x0000,0x0000,0x0000,0x0000,0x0000 }, /* data rate=10K */ 	
   {0x0016,0x00CD,0x0000,0x0000,0x0310,0x0000,0x0000,0x0000,0x0000,0x0000 }, /* data rate=25K */	
   {0x0006,0x004C,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000 }, /* data rate=50K */ 	
   {0x0006,0x0119,0x001D,0x0346,0x0022,0x0331,0x0386,0x0012,0x0008,0x0008 }, /* data rate=125K */ 	
   {0x0006,0x0294,0x02CA,0x0062,0x0358,0x03E9,0x03B3,0x003E,0x03E9,0x0039 }  /* data rate=250K */   
};

uc32 *data_rate_index[1] =
{
	(uc32 *)&data_rate_Table16,
};

uc8 *MOD_Register_index[1] =
{
   (uc8 *)&MOD_RegisterTable16,   
};

uc8 *DM_Register_index[1] =
{
   (uc8 *)&DM_RegisterTable16,   
};

uc16 *FCF_Register_index[1] = 
{
   (uc16 *)&FCF_RegisterTable16,      
};

#define	_RXG_VALUE_		0x8F		//Gain=80dB
uc8 AGC_RegisterTable[][2] = 	
{ {AGC_CTL2_REGS,0x40},{AGC_CTL3_REGS,0x24},{AGC_CTL5_REGS,0x00},{AGC_CTL7_REGS,0x30} };

/*****************************************************************************************************
* @brief  BC3602 goto default mode.
* @retval None
*******************************************************************************************************/
void goto_default_mode(void)
{
	u8	x;
	BC3602_LightSleepMode();
#if	( _DEFAULT_MODE_ == _DEEP_SLEEP_MODE_ )
	BC3602_DeepSleepMode();	
#elif	( _DEFAULT_MODE_ == _LIGHT_SLEEP_MODE_ )
	BC3602_LightSleepMode();	
#elif	( _DEFAULT_MODE_ == _IDLE_MODE_ )
	BC3602_IdleMode();	
#endif

#if	( _LIRC_ENABLE_ == 0)
	x  = BC3602_ReadRegister(LIRC_CTL_REGS);
	BC3602_WriteRegister(LIRC_CTL_REGS, x & ~0x01);
#endif
}

/*****************************************************************************************************
* @brief  RF parameter initialization.
* @retval None
*******************************************************************************************************/
void parameter_initialization(void)
{
	memcpy((void *)TxPayloadData, (void *)pn9_data, DEFAULT_PKT_Length);
	memcpy((void *)RxPayloadData, (void *)pn9_data, DEFAULT_PKT_Length);
	
	/* set BC3602 bank2 register */	 
	BC3602_AnalogRegisterConfigure();	
	
	/* set Crystal */
	BC3602_CrystalConfigure();
	
	/* set AGC */
	BC3602_AgcConfigure(Enable);
	
	/* set tx and rx preamble */
	BC3602_PreambleConfigure(DEFAULT_TX_Preamble,DEFAULT_RX_Preamble);

	/* set Syncword */
	BC3602_SyncwordConfigure(DEFAULT_SyncWidth, DEFAULT_DeviceID);

	/* set tx packet length */
	BC3602_SetTxPayloadWidth(DEFAULT_PKT_Length);   
	
	/* set rx packet length */
	BC3602_SetRxPayloadWidth(DEFAULT_PKT_Length);   

  /* set header type */
	BC3602_HeaderConfigure(DEFAULT_PLLEN_EN, DEFAULT_PLHAC_EN, DEFAULT_PLHLEN, DEFAULT_PLH_EN); 

	/* set codeing type */
	BC3602_ManchesterConfigure(DEFAULT_Man_EN);
	BC3602_FecConfigure(DEFAULT_FEC_EN);
  BC3602_Crc_Configure(DEFAULT_CRC_EN, DEFAULT_CRCFMT);
  BC3602_TrailerConfigure(DEFAULT_Tra_EN); 

	/* set whitening type */
	BC3602_WhiteningConfigure(DEFAULT_WHT_EN, DEFAULT_WHTFMT, DEFAULT_WHTSEED);

	/* set RF band */
  BC3602_FrequencyConfigure(DEFAULT_RF_Frequency);

  /* set Tx power */
  BC3602_PowerConfigure(DEFAULT_TX_Power);
  
  /* set data rate */
  BC3602_DataRateConfigure(DEFAULT_DATA_RATE);
   
	BC3602_CrystalReady();	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 IC Initial                                          */
/*----------------------------------------------------------------------------*/
void  BC3602_Initial(void)
{
	u8 x;
	BC3602_InterfaceConfigure();  
  BC3602_LightSleepMode();
	/* 1.2V RSTLL */
	x = BC3602_ReadRegister(CLOCK_CTL_REGS);
  BC3602_WriteRegister(CLOCK_CTL_REGS, x | 0x01);
  BC3602_WriteRegister(CLOCK_CTL_REGS, x & ~0x01);
	x = (BC3602_ReadRegister(GIO12_CTL_REGS) & ~0xC0);		//PAD[7:6] drive strength 0:0.5mA, 1:1mA, 2:5mA, 3:10mA
	x |= _GIO_DRIVING_;													
	BC3602_WriteRegister(GIO12_CTL_REGS, x);
	 
	parameter_initialization();
	
	BC3602_VcoCalibartion();           
  BC3602_LircCalibartion();   	 

	tx_packet_time = cal_packet_time(_TX_SEL_);
	rx_packet_time = cal_packet_time(_RX_SEL_);
	goto_default_mode();
}
/*----------------------------------------------------------------------------*/
/*	 MCU & BC3602 interface Configure                                          */
/*----------------------------------------------------------------------------*/
void BC3602_InterfaceConfigure(void)
{
   RF_SPI_CFGR &= ~((15UL << (_RF_CSN_*4))  |
                    (15UL << (_RF_SCK_*4))  |
                    (15UL << (_RF_SDIO_*4)) );      
   RF_SPI_PORT->DIRCR |= (RF_CSN | RF_SCK | RF_SDIO); 	/* CSN/SCK/SDIO for output mode */
   RF_SPI_PORT->ODR &= ~(RF_CSN | RF_SCK | RF_SDIO);  	/* CMOS output */
   RF_SPI_PORT->INER |= RF_SDIO;                      /* input enable */
   RF_SPI_PORT->PUR |= RF_SDIO;                       /* pull-high enable */
  #if (_IRQ_ENABLE_ == 1)   
   RF_IO_CFGR &= ~(15UL << (_RF_IRQ_*4));              /* RF_IRQ to GPIO mode */   
   RF_IO_PORT->DIRCR &= ~RF_IRQ;                      /* IRQ for input mode */
   RF_IO_PORT->INER |= RF_IRQ;                        /* input enable */
   RF_IO_PORT->PUR |= RF_IRQ;                         /* pull-high enable */
  #endif   
   RF_CSN_HIGH;
   RF_SCK_LOW;
   RF_SDIO_LOW;

#if (_SPI_ARCHITECTURE_ != 0)
	/* SPI Configuration */	
   RF_SPI->CR1 =	SPI_MASTER |  SPI_DATALENGTH_8 |
                  SPI_SEL_SOFTWARE | SPI_SELPOLARITY_LOW |
                  SPI_FIRSTBIT_MSB | 0x0100;			/* CPOL=1, CPHA=1 */
   /*-- Enable or Disable the specified SPI interrupt --*/
   RF_SPI->IER = 0x00000000;
   /*-- SPIx FIFO Control Register Configuration --*/
   RF_SPI->FCR = SPI_FIFO_DISABLE;
   /*-- SPIx Clock Prescaler Register Configuration -*/
   RF_SPI->CPR = ((CKCU_GetPeripFrequency(CKCU_PCLK_SPI1)/_SPI_SPEED_)-1)/2; /* fSCK=fPCLK/(2*(CP + 1)) */
   /* Enable or Disable the SEL output for the specified SPI peripheral.*/
   RF_SPI->CR0 = 0x00000008;
   /* Enable the selected SPI peripheral */
   RF_SPI->CR0 |= 0x00000001;	
   /* configure CSN/SCK/MOSI/MISO as GPIO  */   
   AFIO_GPxConfig(GPIO_PA,RF_CSN|RF_SCK|RF_MOSI|RF_MISO,AFIO_FUN_SPI);	
#endif	

   BC3602_LightSleepMode();
   /* Software reset BC3602 */
   BC3602_StrobeCommand(SOFT_RESET_CMD);
   /* register bank change to 0 */
   BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);
   BC3602_WriteRegister(GIO12_CTL_REGS,_GPIO_CURRENT_ | (_GIO2_FUNCTION_ << 3) | _GIO1_FUNCTION_ );
   BC3602_WriteRegister(GIO34_CTL_REGS,(_GIO4_FUNCTION_ << 4) | _GIO3_FUNCTION_ );
   BC3602_WriteRegister(GPIO_PULL_UP_REGS,_SPI_PULL_HIGH_ | 0x01);

#if (_IRQ_ENABLE_ == 1)
   BC3602_GioConfigure(_IRQ_LINE_, INT_REQUEST);
   BC3602_WriteRegister(IRQ_CTL_REGS,0x02 | _BC3602_REGS_IRQ1_ | _IRQ_POLARITY_ );
	 BC3602_WriteRegister(IRQ_STATUS_REGS, 0xFF);		//clear IRQ request
#endif   
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 LIRC calibration                                                   */
/*----------------------------------------------------------------------------*/
void BC3602_LircCalibartion(void)
{
   BC3602_WriteRegister(LIRC_CTL_REGS,BC3602_ReadRegister(LIRC_CTL_REGS) | 0x80);
   while(BC3602_ReadRegister(LIRC_CTL_REGS) & 0x80);   
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 VCO/RC calibration                                                */
/*----------------------------------------------------------------------------*/
void BC3602_VcoCalibartion(void)
{
   BC3602_WriteRegister(OPERATION_CTL_REGS,BC3602_ReadRegister(OPERATION_CTL_REGS) | 0x08);
   while(BC3602_ReadRegister(OPERATION_CTL_REGS) & 0x08);  
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 VCO/RC calibration                                                */
/*----------------------------------------------------------------------------*/
void BC3602_CrystalReady(void)
{
	u8 cc;
	do
	{
		cc = BC3602_ReadRegister(CLOCK_CTL_REGS);
	}
   while(!(cc & 0x20));
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 GPIO interface Configure                                           */
/*----------------------------------------------------------------------------*/
void BC3602_GioConfigure(u8 gio,u8 fun)
{
   u8 regs,value;
   regs = GIO12_CTL_REGS + (gio / 2 );
   value = BC3602_ReadRegister(regs);
   if(gio < 2 )
   {
      value &= ~(0x07 << (3*(gio % 2)));
      value |= ((fun & 0x07) << (3*(gio % 2)));
   }
   else
   {
      value &= ~(0x0F << (4*(gio % 2)));
      value |= ((fun & 0x0F) << (4*(gio % 2)));
   }
   BC3602_WriteRegister(regs,value);
}
 
/*----------------------------------------------------------------------------*/
/*	 BC3602 IRQ Enable Configure                                           */
/*----------------------------------------------------------------------------*/
void BC3602_IrqConfigure(u8 irq_en)
{
	BC3602_WriteRegister(IRQ_ENABLE_REGS,irq_en);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 AGC register configure                                             */
/*----------------------------------------------------------------------------*/
void BC3602_AgcConfigure(u8 en)
{
	u8 x,y;  

	x=BC3602_ReadRegister(CONFIG_REGS);	
	if(en == 1)
		BC3602_WriteRegister(CONFIG_REGS, x | 0x40);
	else
		BC3602_WriteRegister(CONFIG_REGS, x & 0xBF);
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK2);            /* register bank 2 */  
	BC3602_WriteRegister(GAIN_CTL_REGS,_RXG_VALUE_);
	x = sizeof(AGC_RegisterTable)/2;   
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK1);            /* register bank 1 */   
	for(y=0;y<x;y++)
	{
		BC3602_WriteRegister(AGC_RegisterTable[y][0],AGC_RegisterTable[y][1]);
	}
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 crystal_configure                                                */
/*----------------------------------------------------------------------------*/
void BC3602_CrystalConfigure(void)
{
	/* set crystal type */
	BC3602_WriteRegister(XO_SEL_CTL_REGS,0x03);				//16MHz
	
	/* set crystal TRIM */
	BC3602_WriteRegister(XO_CAP_CTL_REGS,0x41);				//For BCM-3602-X01
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 preamble_configure                                                */
/*----------------------------------------------------------------------------*/
void  BC3602_PreambleConfigure(u16 tx_preamble,u8 rx_preamble)
{
	u8 x;
	BC3602_WriteRegister(PREAMBLE_LENG_REGS,tx_preamble-1);
	/* set rx preamble */
  x = BC3602_ReadRegister(PACKET_CTL2_REGS) & 0xFC;
	x = x + rx_preamble_value[rx_preamble];
	//x |= sync_width_value[(*BC3602_T).sync_length]; 
  BC3602_WriteRegister(PACKET_CTL2_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 syncword_configure                                                */
/*----------------------------------------------------------------------------*/
void  BC3602_SyncwordConfigure(u8 length, int64_t id)
{
	u8 x;
	
	/* set syncword length */
  x = BC3602_ReadRegister(PACKET_CTL2_REGS) & 0xF3;
	x |= sync_width_value[length];
  BC3602_WriteRegister(PACKET_CTL2_REGS,x);
	syncword_calc(32, id, sync_word_buff);
	syncword_calc(32, id >> 20, sync_word_buff+4);

	BC3602_WriteSyncWord((u8 *)sync_word_buff,8);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 header configure      
				PLL_EN:			payload_length_field_EN
				PLHAC_EN:		payload header address correction enable control
				PLHLEN:			payload header_length
				PLH_EN:			payload header_field_EN
*/
/*----------------------------------------------------------------------------*/
void  BC3602_HeaderConfigure(u8 PLLEN_EN, u8 PLHAC_EN, u8 PLHLEN, u8 PLH_EN)
{
	u8 x;
	x = BC3602_ReadRegister(PACKET_CTL3_REGS) & 0xF0;
	x = x + (PLLEN_EN <<3) + (PLHAC_EN <<2) + (PLHLEN <<1) + (PLH_EN);
	BC3602_WriteRegister(PACKET_CTL3_REGS,x);
	
	/* set PKT8 address */
	x = BC3602_ReadRegister(PACKET_CTL3_REGS) & 0x04;
	if(x == 0x04)
		BC3602_WriteRegister(HEADER_ADDR0_REGS,0x3F);
	else
		BC3602_WriteRegister(HEADER_ADDR0_REGS,0x35);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 manchester configure         
				man_EN:			Manchester code enable					
*/
/*----------------------------------------------------------------------------*/
void  BC3602_ManchesterConfigure(u8 man_EN)
{
	u8 x;
	x = BC3602_ReadRegister(PACKET_CTL3_REGS) & 0x7F;	
	x = x + (man_EN<<7);
	BC3602_WriteRegister(PACKET_CTL3_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 FEC configure         
				fec_EN:			FEC enable					
*/
/*----------------------------------------------------------------------------*/
void  BC3602_FecConfigure(u8 fec_EN)
{
	u8 x;
	x = BC3602_ReadRegister(PACKET_CTL3_REGS) & 0xBF;	
	x = x + (fec_EN	<<6);	
	BC3602_WriteRegister(PACKET_CTL3_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 CRC configure         
				crc_EN:			CRC field enable
				crcfmt:			CRC format selection					
*/
/*----------------------------------------------------------------------------*/
void  BC3602_Crc_Configure(u8 crc_EN, u8 crcfmt)
{
	u8 x;
	x = BC3602_ReadRegister(PACKET_CTL3_REGS) & 0xCF;	
	x = x + (crc_EN <<5) + (crcfmt <<4);	
	BC3602_WriteRegister(PACKET_CTL3_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 Trailer configure         
				tra_EN:			Trailer enable					
*/
/*----------------------------------------------------------------------------*/
void  BC3602_TrailerConfigure(u8 tra_EN)
{
	u8 x;
	
	x = BC3602_ReadRegister(PACKET_CTL2_REGS) & 0xDF;
	x = x + (tra_EN	<<5);	
	BC3602_WriteRegister(PACKET_CTL2_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 whitening configure				
				wht_EN:			Data whitening enable
				whtfmt:			whitening format selection
				wht_seed:		whitening Seed selection	
*/
/*----------------------------------------------------------------------------*/
void  BC3602_WhiteningConfigure(u8 wht_EN, u8 whtfmt, u8 wht_seed)
{
	u8 x;
	
	x = BC3602_ReadRegister(PACKET_CTL2_REGS) & 0xEF;
	x = x + (whtfmt<<4);	
	BC3602_WriteRegister(PACKET_CTL2_REGS,x);
	
	x = BC3602_ReadRegister(PACKET_CTL4_REGS);
	x = (wht_EN<<7) + wht_seed;	
	BC3602_WriteRegister(PACKET_CTL4_REGS,x);
	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 frequency configure                                                */
/*	 D_N = (INT)(Frequ * N)/XCLK                                               */
/*	 D_K = (INT)((Frequ * N)/XCLK - D_N) * 2^20                                */
/*                                      */
/*----------------------------------------------------------------------------*/
void BC3602_FrequencyConfigure(float frequ)
{
	float fn;
  u8 dn;
  u32 dk;
  if(frequ < 400.0)
  {
     dn = 0x00;
     fn = (frequ * 1000000) * 1;      /* 315MHZ */
  }
  else
  {
     if(frequ >= 800.0) 
     {
        dn = 0x60;
        fn = (frequ * 1000000) * 1;  /* 868/915MHZ */
     }
     else
     {
        fn = (frequ * 1000000) * 1;      /* 433/470/510MHZ */
        if(frequ > 450.0) dn = 0x40;     /* 470~510M band */         
        else  dn = 0x20;                 /* 433M band */
     }
  }
  /* set RF band */
  dn |= BC3602_ReadRegister(OPERATION_CTL_REGS) & 0x9F;
  BC3602_WriteRegister(OPERATION_CTL_REGS,dn);
  /* set RF Frequency */
  fn /= (float)16000000;									//16MHz
  dn = (u8)fn;
  fn -= (float)dn;
  fn *= (float)(1048576.0);
  dk = (u32)fn;
  BC3602_WriteRegister(FRACTIONAL_N_DN_REGS,dn);
  BC3602_WriteRegister(FRACTIONAL_N_DKL_REGS,dk & 0xFF);
  BC3602_WriteRegister(FRACTIONAL_N_DKM_REGS,(dk >> 8) & 0xFF);
  BC3602_WriteRegister(FRACTIONAL_N_DKH_REGS,(dk >>16) & 0xFF); 
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 output power configure                                             */
/*----------------------------------------------------------------------------*/
void BC3602_PowerConfigure(u8 pwr)
{
	u8 power;
	power = rf_power_value[DEFAULT_RF_Band][pwr];
  BC3602_WriteRegister(TX_POWER_CTL_REGS,power);   /* TX control register 2 */
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 data rate configure                                                */
/*	 Data Rate Register(DTR) = ((XCLK / 32) / data rate) - 1                   */
/*----------------------------------------------------------------------------*/
void BC3602_DataRateConfigure(u8 dr)
{ 
	void *idx;
  u8 i,x;
	uc8 *idxx;
   
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */

	// if(data rate < 100K ) IF = 200K else IF = 300K/
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK2);            /* register bank 2 */
  if(dr < 5)  // < 100K 									// IF frequency = 200K //
  {  					 			
		BC3602_WriteRegister(RX2_CTL_REGS,0x44);										//  20190423   popo  0x08; 
	}          	
  else 															// IF frequency = 300K //
  {    
		BC3602_WriteRegister(RX2_CTL_REGS,0x54); 						//  20190423   popo  0x18;		
	}
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
	
	/* MOD configure */
  idx = (void *)MOD_Register_index[0]; 		//Crystal = 16MHz
  BC3602_WriteMultibyteRegister(WRITE_REGS_CMD+MODULATOR_CTL1_REGS,(u8 *)((uc8 *)idx+dr*8),3);

  /* RBCLK &  De-modulator configure */
  idx = (void *)DM_Register_index[0]; 		//Crystal = 16MHz   
  BC3602_WriteMultibyteRegister(WRITE_REGS_CMD+DEMOULATOR_CTL1_REGS,(u8 *)((uc8 *)idx+dr*8),8);	
	 
  /* Filter_Coefficient configure */
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK1);            /* register bank 1 */ 
  idx = (void *)FCF_Register_index[0]; 		//Crystal = 16MHz     
  BC3602_WriteRegister(FILTER_CTL1_REGS,*((uc16 *)idx + dr*10) & 0xFF);
  BC3602_WriteMultibyteRegister(WRITE_REGS_CMD+FILTER_CTL2_REGS,(u8 *)((uc16 *)idx + dr*10 + 1),18);
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */

	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK2);            /* register bank 2 */   

	x = (BC3602_ReadRegister(OPERATION_CTL_REGS) >> 5) & 0x03;
	idxx = Analog_Register_index[x];
  x = sizeof(Analog_RegisterTable433)/2;
  for(i=0;i<2;i++)
  {
     BC3602_WriteRegister(*(idxx+i*2+0),*(idxx+i*2+1));
  }  
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */  
	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 Extend_Margin_configure                                                 */
/*----------------------------------------------------------------------------*/
void  BC3602_ExtendMarginConfigure(u8 mar_EN, u8 length)
{
	u8 x;
	x = (BC3602_ReadRegister(FIFO2_CTL_REGS) & 0xF3) | (mar_EN << 2) | length;	//
	BC3602_WriteRegister(FIFO2_CTL_REGS,x);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 register configure                                                 */
/*----------------------------------------------------------------------------*/
void BC3602_AnalogRegisterConfigure(void)
{
  u8 x,i;
  uc8 *idx;
//  Analog_RegisterTable433[][2]
  x = (BC3602_ReadRegister(OPERATION_CTL_REGS) >> 5) & 0x03;
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK2);            /* register bank 2 */   
	
  idx = Analog_Register_index[x];
  x = sizeof(Analog_RegisterTable433)/2;
  for(i=0;i<x;i++)
  {
     BC3602_WriteRegister(*(idx+i*2+0),*(idx+i*2+1));
  }  
 
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */  
	BC3602_VcoCalibartion(); 	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 get data rate                                               			*/
/*----------------------------------------------------------------------------*/
u32 BC3602_GetDatarate(u8 dr)
{
	void *idx;  
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
  idx = (void *)data_rate_index[0]; 		//Crystal = 16MHz
	return *((uc32 *)idx+dr); 
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 enable direct TX mode                                       			*/
/*----------------------------------------------------------------------------*/
void BC3602_EnableDirTx(void)
{
	u8	cc;
	BC3602_LightSleepMode();
  BC3602_WriteRegister(CONFIG_REGS,BC3602_ReadRegister(CONFIG_REGS) | 0x10);
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */   
	cc = BC3602_ReadRegister(OPERATION_CTL_REGS) & 0xFD;			//b2=RTX_EN, b1=RTX_SEL, b0=SX_EN
	cc |= 0x03;
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc);	
	delay_10us(30);
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc | 0x04);	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 enable direct RX mode                                       			*/
/*----------------------------------------------------------------------------*/
void BC3602_EnableDirRx(void)
{
	u8	cc;
	BC3602_LightSleepMode();
  BC3602_WriteRegister(CONFIG_REGS,BC3602_ReadRegister(CONFIG_REGS) | 0x10);
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */   
	cc = BC3602_ReadRegister(OPERATION_CTL_REGS) & 0xFD;			//b2=RTX_EN, b1=RTX_SEL, b0=SX_EN
	cc |= 0x01;
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc);	
	delay_10us(30);
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc | 0x04);	
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 disable direct RX mode                                       			*/
/*----------------------------------------------------------------------------*/
void BC3602_DisableDir(void)
{
	u8	cc;
  BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */   
	cc = BC3602_ReadRegister(OPERATION_CTL_REGS) & ~0x04;		//b2=RTX_EN, b1=RTX_SEL, b0=SX_EN
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc);	
	delay_10us(30);
	cc = BC3602_ReadRegister(OPERATION_CTL_REGS) & ~0x01;		//b2=RTX_EN, b1=RTX_SEL, b0=SX_EN
	BC3602_WriteRegister(OPERATION_CTL_REGS, cc);	

   BC3602_WriteRegister(CONFIG_REGS,BC3602_ReadRegister(CONFIG_REGS) & ~0x10);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 stroble command                                                   */
/*----------------------------------------------------------------------------*/
void BC3602_StrobeCommand(u8 cmd)
{ 
#if (_SPI_ARCHITECTURE_ == 0) 	
   RF_CSN_LOW;
   BC3602_serial_output(cmd);
   RF_CSN_HIGH;    
#else
	 RF_SPI->CR1 |= SPI_DATALENGTH_8;    /* data length=8bits */
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = cmd;	
//   cmd = RF_SPI->DR;                   /* clear SPI RX buffer */  
   while(!(RF_SPI->SR & SPI_FLAG_TXE));
   RF_SPI_INACTIVE;
   while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
   /* clear SPI RX buffer */   
   cmd = RF_SPI->DR;
#endif	
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 read register                                                     */
/*----------------------------------------------------------------------------*/
u8 BC3602_ReadRegister(u8 regs)
{
   u8 value;  
   regs &= REGSADDR_MASK;
   regs |= READ_REGS_CMD;  
#if (_SPI_ARCHITECTURE_ == 0) 	
   RF_CSN_LOW;
   BC3602_serial_output(regs);
   value = BC3602_serial_input();
   RF_CSN_HIGH;
   RF_SDIO_DIRO;   
#else 
   RF_SPI->CR1 &= ~SPI_DATALENGTH_8;    	/* data length=16bits */   
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = (regs << 8) + 0x00FF;
   while(!(RF_SPI->SR & SPI_FLAG_TXE));
   RF_SPI_INACTIVE;   
   while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
   value = (u8)(RF_SPI->DR & 0xFF);
#endif	
	
   return (value);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 write register                                                     */
/*----------------------------------------------------------------------------*/
void BC3602_WriteRegister(u8 regs,u8 data)
{   
   regs &= REGSADDR_MASK;
   regs |= WRITE_REGS_CMD;     
#if (_SPI_ARCHITECTURE_ == 0)	
   RF_CSN_LOW;
   BC3602_serial_output(regs);
   BC3602_serial_output(data);      
   RF_CSN_HIGH; 
#else
	 RF_SPI->CR1 &= ~SPI_DATALENGTH_8;    	/* data length=16bits */
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = (regs << 8) + data;
   while(!(RF_SPI->SR & SPI_FLAG_TXE));
   RF_SPI_INACTIVE;   
   while(!(RF_SPI->SR & SPI_FLAG_RXBNE)); 
   /* clear SPI RX buffer */   
   data = RF_SPI->DR;
	
#endif	
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 read multibyte register                                            */
/*----------------------------------------------------------------------------*/
void BC3602_ReadMultibyteRegister(u8 cmd,u8 *pbuf,u16 length)
{
#if (_SPI_ARCHITECTURE_ == 0)                   
   RF_CSN_LOW;
   BC3602_serial_output(cmd);
   while(length --)
   {
      *pbuf++ = BC3602_serial_input();
   }
   RF_CSN_HIGH;
   RF_SDIO_DIRO;    
#else  
	 RF_SPI->CR1 |= SPI_DATALENGTH_8;    	/* data length=8bits */      
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = cmd;
	while(!(RF_SPI->SR & SPI_FLAG_TXBE));	
	while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
   cmd = RF_SPI->DR;             			/* clear SPI RX buffer */
  while(length--)
   {
      RF_SPI->DR = 0xFF;
  while(!(RF_SPI->SR & SPI_FLAG_TXBE));
      while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
      *pbuf++ = RF_SPI->DR;
   }
   RF_SPI_INACTIVE;
#endif
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 write multibyte register                                           */
/*----------------------------------------------------------------------------*/
void BC3602_WriteMultibyteRegister(u8 cmd,u8 *pbuf,u16 length)
{ 
#if (_SPI_ARCHITECTURE_ == 0)     	
   RF_CSN_LOW;
   BC3602_serial_output(cmd);
   while(length --)
   {
      BC3602_serial_output(*pbuf++);
   }
   RF_CSN_HIGH;
#else
	 RF_SPI->CR1 |= SPI_DATALENGTH_8;    		/* data length=8bits */
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = cmd;
   while(length --)
   {
      while(!(RF_SPI->SR & SPI_FLAG_TXBE));
      RF_SPI->DR = *pbuf++;
   }
   cmd = RF_SPI->DR;                      /* clear SPI RX buffer */
   while(!(RF_SPI->SR & SPI_FLAG_TXE));
   RF_SPI_INACTIVE;   
   while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
   /* clear SPI RX buffer */   
   while(RF_SPI->SR & SPI_FLAG_RXBNE) cmd = RF_SPI->DR;
	 
#endif
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 write multibyte register by ring                                          */
/*----------------------------------------------------------------------------*/
void BC3602_WriteMultibyteRingRegister(u8 cmd,u8 *pbuf,u16 length,u8 buf_st,u8 buf_l)
{      
#if (_SPI_ARCHITECTURE_ == 0)   	
   RF_CSN_LOW;
   BC3602_serial_output(cmd);
   while(length --)
   {
      BC3602_serial_output(*(pbuf+buf_st));
		if(++buf_st >= buf_l)
			buf_st = 0;
   }
   RF_CSN_HIGH;
#else
	 RF_SPI->CR1 |= SPI_DATALENGTH_8;    		/* data length=8bits */
   /* enable CS */
   RF_SPI_ACTIVE;
   RF_SPI->DR = cmd;
   while(length --)
   {
      while(!(RF_SPI->SR & SPI_FLAG_TXBE));
      RF_SPI->DR = *(pbuf+buf_st);
		  if(++buf_st >= buf_l)
				buf_st = 0;
   }
   cmd = RF_SPI->DR;                      /* clear SPI RX buffer */
   while(!(RF_SPI->SR & SPI_FLAG_TXE));
   RF_SPI_INACTIVE;   
   while(!(RF_SPI->SR & SPI_FLAG_RXBNE));
   /* clear SPI RX buffer */   
   while(RF_SPI->SR & SPI_FLAG_RXBNE) cmd = RF_SPI->DR;
	 
#endif
}

#if (_SPI_ARCHITECTURE_ == 0)
/*----------------------------------------------------------------------------*/
/*	 BC3602 3-wrie SPI serial data output                                      */
/*----------------------------------------------------------------------------*/
static void BC3602_serial_output(u8 data)
{
   u8 x;
   RF_SDIO_DIRO;
   for(x=0;x<8;x++)
   {
      RF_SCK_LOW;            
     	if(data & 0x80) RF_SDIO_HIGH; else RF_SDIO_LOW;
      RF_SCK_HIGH;
      data <<= 1;
   }
   __nop();
   __nop();   
   RF_SCK_LOW;         
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 3-wrie SPI serial data input                                       */
/*----------------------------------------------------------------------------*/
static u8 BC3602_serial_input(void)
{
   u8 x,d;
   d = 0;
   RF_SDIO_DIRI;
   for(x=0;x<8;x++)
   {
      d <<= 1;
      RF_SCK_HIGH;
     if(RF_SDIO_IN) d |= 0x01;
      RF_SCK_LOW;
   }
   return d;
}
#endif
/*****************************************************************************************************
* @brief  Is the two buffer data same ?
* @retval None
*******************************************************************************************************/
bool is_same_data(u8 *saddr,u8 *taddr,u32 len)
{
	u8	i;
	for(i=0; i<len; i++)
	{
		if(*(taddr+i) != *(saddr+i))
			return FALSE;
	}
	return TRUE;
}
/*****************************************************************************************************
* @brief  Is the two buffer data same by ring ?
* @retval None
*******************************************************************************************************/
bool is_same_data_ring(u8 *addr1,u8 *addr2,u32 len,u8 str,u8 tlen)
{
	u8	i, cc;
	for(i=0; i<len; i++)
	{
		cc = *(addr2 + str);
		if(++str >= tlen)
			str = 0;
		if(*(addr1+i) != cc)
			return FALSE;
	}
	return TRUE;
}

/*****************************************************************************************************
* @brief  tx carrier mode process.
* @retval None
*******************************************************************************************************/
u32 cal_packet_time(u8 sel)
{
	
	// (*BC3602_T).
	u32	time, dr;
	u8	c;
 	time = DEFAULT_PKT_Length;
	c = BC3602_ReadRegister(PACKET_CTL3_REGS);
	if((c & 0x03) == 0x01) time++;						//PLH_EN=1
	else if((c & 0x03) == 0x03) time += 2;				//PLHLEN=1 & PLH_EN=1
	if(c & 0x08) time++;										//PLHAC_EN=1
	if(c & 0x20) time += 2;									//CRC_EN=1
	if(c & 0x40) time = (time * 7 + 3) / 4; 	      //*1.75	FEC_EN=1
	if(c & 0x80) time *= 2;									//*2		MCH_EN=1
	if(sel == _TX_SEL_)
		time += DEFAULT_TX_Preamble;
	else		
	{
		time += rx_preamble_value[DEFAULT_RX_Preamble];
		if(rx_preamble_value[DEFAULT_RX_Preamble] == 3)
			time++;
	}		
	time +=  (sync_width_value[DEFAULT_SyncWidth] >> 1) + 2;
	time = time * 8 + 4 + 8;	//4 trailer & 8 dummy clock
	dr = BC3602_GetDatarate(DEFAULT_DATA_RATE);
	dr = 1000000 / dr;
	time *= dr;
	return time;
}


/////////////////////////////////////////////////////////////////////////////
//                               Layer2
/////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------*/
/*	 BC3602 auto cycle timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetAutoCycleMode(u8 md,bool en)
{
   u8 cnt_reg; 
   cnt_reg = BC3602_ReadRegister(ATR_CONTROL_REGS) & 0xF8;
   cnt_reg |= ((md & 0x03) << 1) | en;
   BC3602_WriteRegister(ATR_CONTROL_REGS,cnt_reg);
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
u8	BC3602_SetCycleClockUnit(u32 tm)
{
	u8		cc;
	cc = BC3602_ReadRegister(ATR_CONTROL_REGS) & ~0xC0;	
	if(tm >= 16000000)
		cc |= 0xC0;
	else if(tm >= 8000000)
		cc |= 0x80;
	else if(tm >= 2000000)
		cc |= 0x40;
	BC3602_WriteRegister(ATR_CONTROL_REGS, cc);
	return	cc;
}
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
u16	BC3602_CalCycleCounter(u32 tm)
{
   float clk_tim;
   u16   cyc_value;
	 u8		cc;
	 cc = BC3602_ReadRegister(ATR_CONTROL_REGS);	
   switch(cc & 0xC0)
   {
      case 0x00 : clk_tim = (float)1000000.0/32768.0; break;
      case 0x40 : clk_tim = (float)1000000.0/8192.0; break;
      case 0x80 : clk_tim = (float)1000000.0/4096.0; break;
      case 0xC0 : clk_tim = (float)1000000.0/2048.0; break;
   }   
   clk_tim = (float)(tm-30) / clk_tim;
   cyc_value = (u16)(clk_tim);
	return cyc_value;
}

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void	BC3602_EnableATRCTM(bool flag)
{
	u8		cc;
	cc = BC3602_ReadRegister(ATR_CONTROL_REGS);	
	if(flag)
		cc |= 0x08;
	else
		cc &= ~0x08;
	BC3602_WriteRegister(ATR_CONTROL_REGS, cc);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 auto cycle timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetAutoCycleTimer(u32 tm)
{
   u16   cyc_value;
	 BC3602_SetCycleClockUnit(tm);
	 cyc_value = BC3602_CalCycleCounter(tm);
   BC3602_WriteMultibyteRegister(ATR_CYCLE_L_REGS+WRITE_REGS_CMD,(u8 *)&cyc_value,2);
}

/*----------------------------------------------------------------------------*/
/*	 BC3602 tune cycle counter																*/
/*----------------------------------------------------------------------------*/
void BC3602_TuneAutoCycleCounter(int16_t cnt)
{
   u32   cyc_counter;
	 cyc_counter = BC3602_GetAutoCycleCounter();
	 cyc_counter +=	cnt;
   BC3602_WriteMultibyteRegister(ATR_CYCLE_L_REGS+WRITE_REGS_CMD,(u8 *)&cyc_counter, 2);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 ATRCT timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetATRCT_Timer(u32 tm)
{
   u16   cyc_value;
	 cyc_value = BC3602_CalCycleCounter(tm);
   BC3602_WriteMultibyteRegister(ATRCT_L_REGS+WRITE_REGS_CMD,(u8 *)&cyc_value,2);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 ATRCT timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetATRCT_Counter(u16 atrct)
{
   BC3602_WriteMultibyteRegister(ATRCT_L_REGS+WRITE_REGS_CMD,(u8 *)&atrct,2);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 ATRCT timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
u32 BC3602_GetATRCT_Counter(void)
{
   u16   atrct;
   BC3602_ReadMultibyteRegister(ATRCT_L_REGS+READ_REGS_CMD,(u8 *)&atrct,2);
	 return atrct;
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 auto cycle timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
u16 BC3602_GetAutoCycleCounter(void)
{
   u16 cyc_value;
   BC3602_ReadMultibyteRegister(ATR_CYCLE_L_REGS+READ_REGS_CMD,(u8 *)&cyc_value,2);
   return(cyc_value);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 auto cycle timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetAutoRxActivePeriod(u32 tm)
{
   u16    uart;
	 u8		x;

	x = BC3602_ReadRegister(ATR_CONTROL_REGS);
	if(tm > 255750)
		x |= 0x10;
	else
		x	&= ~0x10;		
	BC3602_WriteRegister(ATR_CONTROL_REGS, x);	
	
	if(x & 0x10)
   	uart = (tm+999) / 1000;
	else
   	uart = (tm+249) / 250;
	if(uart > 0)
   	uart--;
	x = (u8) uart;
   BC3602_WriteRegister(ATR_ACTIVE_REGS, x);
	x = (u8) (uart >> 8);
   BC3602_WriteRegister(ATR_ACTIVE_H_REGS, x);
}
/*----------------------------------------------------------------------------*/
/*	 BC3602 auto cycle timer(unit-->us)                                        */
/*----------------------------------------------------------------------------*/
void BC3602_SetAutoRxExtendPeriod(u32 tm)
{
   u16    uart;
	 u8		x;
	x = BC3602_ReadRegister(ATR_CONTROL_REGS);
	if(x & 0x10)
   	uart = (tm+999) / 1000;
	else
   	uart = (tm+249) / 250;
   uart--;
   BC3602_WriteMultibyteRegister(ATR_EACTIVE_L_REGS+WRITE_REGS_CMD,(u8 *)&uart,2);
}
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
void BC3602_EnableARK(bool en)
{
	u8	x;

	x =  _BC3602_REGS_ARK1_;
	if(en)
		x |= 0x01;
	else
		x &= ~0x01;
	BC3602_WriteRegister(ARK_CONTROL_REGS, x);
	BC3602_WriteRegister(ARK_ACTIVE_REGS, _BC3602_REGS_ARK2_);
}
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
bool BC3602_SetARKRXAP_Timer(u32 tm)
{
   u8		x;
	u16	uart;
	bool	flag = TRUE;									 
	x = BC3602_ReadRegister(ATR_CONTROL_REGS);
	if(tm > 64000)
		x |= 0x10;
	else
		x	&= ~0x10;		
	BC3602_WriteRegister(ATR_CONTROL_REGS, x);	
	
	if(x & 0x10)
   	uart = (tm+999) / 1000;
	else
   	uart = (tm+249) / 250;
	uart--;
	if(uart > 0x00FF)
	{
		flag = FALSE;
		uart = 0x00FF;
	}
	BC3602_WriteRegister(ARK_ACTIVE_REGS, (u8)(uart & 0xFF));

	return flag;
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
u8	BC3602_GetAutoRetryNumber(void)
{
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
	return ((BC3602_ReadRegister(ARK_CONTROL_REGS) >> 4) + 1);
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
void BC3602_SetAutoRetryNumber(u8 counter)						//ARK resend counter  max value 16
{
	u8 x;
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
	x = BC3602_ReadRegister(ARK_CONTROL_REGS) << 4;
	x = (x >> 4) + ((counter - 1) << 4);
	BC3602_WriteRegister(ARK_CONTROL_REGS, x);
	BC3602_StrobeCommand(REGS_BANK_CMD+REGS_BANK0);            /* register bank 0 */
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
void BC3602_IncPID(void)
{
	u8	x;
	x = (BC3602_ReadRegister(PACKET_CTL2_REGS) + 0x40) & 0xFF;
	BC3602_WriteRegister(PACKET_CTL2_REGS, x);
}

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
u8 BC3602_ReadReceiveRSSI(void)
{
	u8	x;
	x = BC3602_ReadRegister(RSSI_VALUE_ID_REGS);
	return x;
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
u8 BC3602_ReadEnvironmentRSSI(void)
{
	u8	x;
	x = BC3602_ReadRegister(RSSI_VALUE_REGS);
	return x;
}
/*----------------------------------------------------------------------------*/
